<?php
$totalBranches = count($branches);
$totalRevenue = array_sum(array_column($branches, 'monthly_revenue'));
$topBranch = !empty($branches) ? $branches[0] : null;
$totalEmployees = array_sum(array_column($branches, 'employee_count'));

$content = '
<!-- Page Header -->
<div class="page-header mb-4">
    <div class="row align-items-center">
        <div class="col">
            <h1 class="page-title">Quản lý chi nhánh</h1>
            <p class="page-subtitle text-muted">Theo dõi hiệu suất và quản lý các chi nhánh</p>
        </div>
        <div class="col-auto">
            ' . ($_SESSION['user_role'] == 'owner' ? '
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addBranchModal">
                <i class="fas fa-plus me-2"></i>Thêm chi nhánh
            </button>' : '') . '
        </div>
    </div>
</div>

<!-- Branch Overview -->
<div class="row mb-4">
    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card border-start-primary">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <div class="text-muted small">Tổng chi nhánh</div>
                        <div class="value text-primary" id="totalBranches">' . $totalBranches . '</div>
                        <div class="trend text-success">
                            <i class="fas fa-store me-1"></i>
                            <span>' . ($branchStats['active_branches'] ?? $totalBranches) . ' đang hoạt động</span>
                        </div>
                    </div>
                    <div class="stat-icon">
                        <i class="fas fa-code-branch"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card border-start-success">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <div class="text-muted small">Doanh thu tháng này</div>
                        <div class="value text-success" id="totalRevenue">' . number_format($totalRevenue, 0, ',', '.') . 'đ</div>
                        <div class="trend text-success">
                            <i class="fas fa-arrow-up me-1"></i>
                            <span>Tháng hiện tại</span>
                        </div>
                    </div>
                    <div class="stat-icon">
                        <i class="fas fa-dollar-sign"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card border-start-warning">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <div class="text-muted small">Chi nhánh hiệu quả nhất</div>
                        <div class="value text-warning" id="topBranch">' . ($topBranch ? $topBranch['name'] : 'N/A') . '</div>
                        <div class="trend text-success">
                            <i class="fas fa-crown me-1"></i>
                            <span>' . ($topBranch ? number_format($topBranch['performance_score'], 1) . '% hiệu suất' : 'N/A') . '</span>
                        </div>
                    </div>
                    <div class="stat-icon">
                        <i class="fas fa-trophy"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card border-start-info">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <div class="text-muted small">Tổng nhân viên</div>
                        <div class="value text-info" id="totalStaff">' . $totalEmployees . '</div>
                        <div class="trend text-success">
                            <i class="fas fa-users me-1"></i>
                            <span>Trung bình ' . ($totalBranches > 0 ? number_format($totalEmployees / $totalBranches, 1) : 0) . '/chi nhánh</span>
                        </div>
                    </div>
                    <div class="stat-icon">
                        <i class="fas fa-users"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Branch Performance Chart -->
    <div class="col-xl-8 mb-4">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">So sánh hiệu suất chi nhánh</h5>
                <div class="chart-controls">
                    <div class="btn-group btn-group-sm" role="group">
                        <input type="radio" class="btn-check" name="chartPeriod" id="thisMonth" autocomplete="off" checked>
                        <label class="btn btn-outline-secondary" for="thisMonth">Tháng này</label>
                        
                        <input type="radio" class="btn-check" name="chartPeriod" id="lastMonth" autocomplete="off">
                        <label class="btn btn-outline-secondary" for="lastMonth">Tháng trước</label>
                        
                        <input type="radio" class="btn-check" name="chartPeriod" id="quarter" autocomplete="off">
                        <label class="btn btn-outline-secondary" for="quarter">Quý này</label>
                    </div>
                </div>
            </div>
            <div class="card-body">
                <div class="chart-container">
                    <canvas id="branchChart"></canvas>
                </div>
            </div>
        </div>
        
        <!-- Branch List -->
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="mb-0">Danh sách chi nhánh</h5>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Chi nhánh</th>
                                <th>Địa chỉ</th>
                                <th>Quản lý</th>
                                <th>Nhân viên</th>
                                <th>Doanh thu tháng</th>
                                <th>Hiệu suất</th>
                                <th>Trạng thái</th>
                                <th>Thao tác</th>
                            </tr>
                        </thead>
                        <tbody>';

if (!empty($branches)) {
    foreach ($branches as $branch) {
        $performanceScore = $branch['performance_score'] ?? 0;
        $performanceColor = $performanceScore >= 80 ? 'success' : ($performanceScore >= 60 ? 'warning' : 'danger');
        $statusColor = $branch['status'] == 'active' ? 'success' : ($branch['status'] == 'maintenance' ? 'warning' : 'secondary');
        $statusText = $branch['status'] == 'active' ? 'Hoạt động' : ($branch['status'] == 'maintenance' ? 'Bảo trì' : 'Tạm dừng');
        
        $content .= '
                            <tr>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="branch-icon me-3">
                                            <i class="fas fa-store text-' . $performanceColor . '"></i>
                                        </div>
                                        <div>
                                            <div class="branch-name">' . htmlspecialchars($branch['name']) . '</div>
                                            <div class="branch-code text-muted small">' . htmlspecialchars($branch['code'] ?? 'N/A') . '</div>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <div class="branch-address">' . htmlspecialchars($branch['address']) . '</div>
                                    <div class="branch-phone text-muted small">' . htmlspecialchars($branch['phone']) . '</div>
                                </td>
                                <td>
                                    <div class="manager-info">
                                        <div class="manager-name">' . htmlspecialchars($branch['managers'] ?? 'Chưa có') . '</div>
                                        <div class="manager-contact text-muted small">' . htmlspecialchars($branch['email'] ?? '') . '</div>
                                    </div>
                                </td>
                                <td>
                                    <div class="staff-count">
                                        <span class="fw-bold">' . ($branch['employee_count'] ?? 0) . '</span> nhân viên
                                        <div class="staff-status text-muted small">' . ($branch['employee_count'] ?? 0) . ' đang làm việc</div>
                                    </div>
                                </td>
                                <td>
                                    <div class="revenue">
                                        <div class="fw-bold text-success">' . number_format($branch['monthly_revenue'] ?? 0, 0, ',', '.') . 'đ</div>
                                        <div class="revenue-growth text-muted small">Tháng này</div>
                                    </div>
                                </td>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="performance-bar me-2">
                                            <div class="progress" style="width: 60px; height: 6px;">
                                                <div class="progress-bar bg-' . $performanceColor . '" style="width: ' . min($performanceScore, 100) . '%"></div>
                                            </div>
                                        </div>
                                        <span class="text-' . $performanceColor . ' fw-bold">' . number_format($performanceScore, 1) . '%</span>
                                    </div>
                                </td>
                                <td>
                                    <span class="badge bg-' . $statusColor . '">' . $statusText . '</span>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <button class="btn btn-outline-primary" onclick="viewBranch(' . $branch['id'] . ')" title="Xem chi tiết">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        ' . (in_array($_SESSION['user_role'], ['owner', 'manager']) ? '
                                        <button class="btn btn-outline-secondary" onclick="editBranch(' . $branch['id'] . ')" title="Chỉnh sửa">
                                            <i class="fas fa-edit"></i>
                                        </button>' : '') . '
                                        <button class="btn btn-outline-info" onclick="viewReport(' . $branch['id'] . ')" title="Báo cáo">
                                            <i class="fas fa-chart-bar"></i>
                                        </button>
                                        ' . ($_SESSION['user_role'] == 'owner' ? '
                                        <button class="btn btn-outline-danger" onclick="deleteBranch(' . $branch['id'] . ')" title="Xóa">
                                            <i class="fas fa-trash"></i>
                                        </button>' : '') . '
                                    </div>
                                </td>
                            </tr>';
    }
} else {
    $content .= '
                            <tr>
                                <td colspan="8" class="text-center py-4">
                                    <div class="text-muted">
                                        <i class="fas fa-store fa-3x mb-3"></i>
                                        <p>Chưa có chi nhánh nào</p>
                                    </div>
                                </td>
                            </tr>';
}

$content .= '
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Branch Analytics & Quick Actions -->
    <div class="col-xl-4 mb-4">
        <!-- Performance Ranking -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">Bảng xếp hạng chi nhánh</h5>
            </div>
            <div class="card-body p-0">
                <div class="ranking-list">';

if (!empty($performanceData)) {
    foreach ($performanceData as $index => $branch) {
        $rankColor = $index === 0 ? 'warning' : ($index === 1 ? 'secondary' : 'info');
        $rankIcon = $index === 0 ? 'crown' : ($index === 1 ? 'medal' : 'award');
        
        $content .= '
                    <div class="ranking-item p-3 ' . ($index < count($performanceData) - 1 ? 'border-bottom' : '') . '">
                        <div class="d-flex align-items-center">
                            <div class="ranking-position me-3">
                                <span class="badge bg-' . $rankColor . '">' . ($index + 1) . '</span>
                            </div>
                            <div class="flex-grow-1">
                                <div class="ranking-name">' . htmlspecialchars($branch['name']) . '</div>
                                <div class="ranking-metrics text-muted small">' . number_format($branch['performance_score'], 1) . '% hiệu suất • ' . number_format($branch['total_revenue'] / 1000000, 1) . 'M doanh thu</div>
                            </div>
                            <div class="ranking-badge">
                                <i class="fas fa-' . $rankIcon . ' text-' . $rankColor . '"></i>
                            </div>
                        </div>
                    </div>';
    }
} else {
    $content .= '
                    <div class="p-3 text-center text-muted">
                        <p>Chưa có dữ liệu hiệu suất</p>
                    </div>';
}

$content .= '
                </div>
            </div>
        </div>
        
        <!-- Quick Actions -->
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">Thao tác nhanh</h5>
            </div>
            <div class="card-body">
                <div class="d-grid gap-2">
                    <button class="btn btn-primary btn-sm" onclick="compareBranches()">
                        <i class="fas fa-balance-scale me-2"></i>So sánh chi nhánh
                    </button>
                    <button class="btn btn-success btn-sm" onclick="exportBranchReport()">
                        <i class="fas fa-download me-2"></i>Xuất báo cáo tổng hợp
                    </button>
                    <button class="btn btn-info btn-sm" onclick="scheduleMeeting()">
                        <i class="fas fa-calendar me-2"></i>Lên lịch họp quản lý
                    </button>
                    <button class="btn btn-warning btn-sm" onclick="sendAlert()">
                        <i class="fas fa-bell me-2"></i>Gửi cảnh báo hiệu suất
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Recent Activities -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Hoạt động gần đây</h5>
            </div>
            <div class="card-body p-0">
                <div class="activity-list">
                    <div class="activity-item p-3 border-bottom">
                        <div class="d-flex">
                            <div class="activity-icon me-3">
                                <i class="fas fa-chart-line text-success"></i>
                            </div>
                            <div class="flex-grow-1">
                                <div class="activity-text">Cập nhật dữ liệu hiệu suất chi nhánh</div>
                                <div class="activity-time text-muted small">Vừa xong</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Branch Modal -->
<div class="modal fade" id="addBranchModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Thêm chi nhánh mới</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="addBranchForm">
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="branchName" class="form-label">Tên chi nhánh</label>
                                <input type="text" class="form-control" id="branchName" name="name" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="branchCode" class="form-label">Mã chi nhánh</label>
                                <input type="text" class="form-control" id="branchCode" name="code" required>
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label for="branchAddress" class="form-label">Địa chỉ</label>
                        <textarea class="form-control" id="branchAddress" name="address" rows="2" required></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="branchPhone" class="form-label">Số điện thoại</label>
                                <input type="tel" class="form-control" id="branchPhone" name="phone" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="branchEmail" class="form-label">Email</label>
                                <input type="email" class="form-control" id="branchEmail" name="email">
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="branchManagers" class="form-label">Quản lý chi nhánh</label>
                                <select class="form-select" id="branchManagers" name="managers[]" multiple size="4">
                                    <option value="">Chọn quản lý (có thể chọn nhiều)</option>';

if (!empty($availableManagers)) {
    foreach ($availableManagers as $manager) {
        $content .= '<option value="' . $manager['id'] . '">' . htmlspecialchars($manager['name']) . ' (' . ucfirst($manager['role']) . ')</option>';
    }
}

$content .= '
                                </select>
                                <small class="text-muted">Giữ Ctrl để chọn nhiều quản lý. Quản lý đầu tiên sẽ là quản lý chính.</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="branchOpenDate" class="form-label">Ngày khai trương</label>
                                <input type="date" class="form-control" id="branchOpenDate" name="open_date" required>
                            </div>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="targetRevenue" class="form-label">Mục tiêu doanh thu (VND)</label>
                                <input type="number" class="form-control" id="targetRevenue" name="target_revenue" value="50000000" min="0">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Hủy</button>
                    <button type="submit" class="btn btn-primary">Thêm chi nhánh</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- View Branch Modal -->
<div class="modal fade" id="viewBranchModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Chi tiết chi nhánh</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="viewBranchContent">
                <div class="text-center py-4">
                    <div class="spinner-border" role="status"></div>
                    <p class="mt-2">Đang tải thông tin...</p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Edit Branch Modal -->
<div class="modal fade" id="editBranchModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Chỉnh sửa chi nhánh</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="editBranchForm">
                <div class="modal-body" id="editBranchContent">
                    <div class="text-center py-4">
                        <div class="spinner-border" role="status"></div>
                        <p class="mt-2">Đang tải thông tin...</p>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Hủy</button>
                    <button type="submit" class="btn btn-primary">Cập nhật chi nhánh</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteBranchModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Xác nhận xóa chi nhánh</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>Bạn có chắc chắn muốn xóa chi nhánh <strong id="deleteBranchName"></strong>?</p>
                <p class="text-danger small">Lưu ý: Chi nhánh sẽ được đánh dấu là đã xóa, không thể khôi phục.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Hủy</button>
                <button type="button" class="btn btn-danger" id="confirmDeleteBranch">Xóa chi nhánh</button>
            </div>
        </div>
    </div>
</div>

<style>
.branch-icon {
    width: 40px;
    height: 40px;
    border-radius: 8px;
    background: #f8f9fa;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.1rem;
}
.branch-name {
    font-weight: 500;
    margin-bottom: 0.25rem;
}
.branch-code {
    font-size: 0.8rem;
}
.branch-address {
    font-size: 0.9rem;
    margin-bottom: 0.25rem;
}
.branch-phone {
    font-size: 0.8rem;
}
.manager-name {
    font-weight: 500;
    margin-bottom: 0.25rem;
}
.manager-contact {
    font-size: 0.8rem;
}
.staff-count .fw-bold {
    font-size: 1.1rem;
}
.staff-status {
    font-size: 0.8rem;
}
.revenue {
    text-align: right;
}
.revenue-growth {
    font-size: 0.8rem;
    font-weight: 500;
}
.performance-bar .progress {
    border-radius: 3px;
}
.ranking-position .badge {
    width: 24px;
    height: 24px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.8rem;
}
.ranking-name {
    font-weight: 500;
    margin-bottom: 0.25rem;
}
.ranking-metrics {
    font-size: 0.8rem;
}
.activity-icon {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background: #f8f9fa;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.9rem;
}
.activity-text {
    font-size: 0.9rem;
    margin-bottom: 0.25rem;
}
.activity-time {
    font-size: 0.8rem;
}

/* FIX NOTIFICATION STYLES */
.toast-container {
    z-index: 9999;
}

.toast {
    background-color: #fff !important;
    border: 1px solid #dee2e6;
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
}

.toast-body {
    color: #212529 !important;
    font-weight: 500;
}

.alert-success {
    background-color: #d1e7dd !important;
    border-color: #badbcc !important;
}

.alert-success .toast-body {
    color: #0f5132 !important;
}

.alert-danger {
    background-color: #f8d7da !important;
    border-color: #f5c2c7 !important;
}

.alert-danger .toast-body {
    color: #842029 !important;
}

.alert-warning {
    background-color: #fff3cd !important;
    border-color: #ffecb5 !important;
}

.alert-warning .toast-body {
    color: #664d03 !important;
}

.alert-info {
    background-color: #cff4fc !important;
    border-color: #b6effb !important;
}

.alert-info .toast-body {
    color: #055160 !important;
}
</style>

<script>
// Global variables
let currentBranchId = null;
let branchChart = null;

// Initialize chart data
const chartData = {
    labels: [],
    revenues: [],
    performances: []
};

// Populate chart data from PHP - FIXED VERSION
';

// PHP code for chart data
if (!empty($performanceData)) {
    $labels = json_encode(array_column($performanceData, 'name'));
    $revenues = json_encode(array_map(function($b) { 
        return round(($b['total_revenue'] ?? 0) / 1000000, 1); 
    }, $performanceData));
    $performances = json_encode(array_column($performanceData, 'performance_score'));
    
    $content .= "
chartData.labels = $labels;
chartData.revenues = $revenues;
chartData.performances = $performances;
";
}

$content .= '

// Branch Performance Chart
function initializeChart() {
    if (document.getElementById("branchChart")) {
        const branchCtx = document.getElementById("branchChart").getContext("2d");
        branchChart = new Chart(branchCtx, {
            type: "bar",
            data: {
                labels: chartData.labels,
                datasets: [{
                    label: "Doanh thu (triệu đồng)",
                    data: chartData.revenues,
                    backgroundColor: "rgba(74, 144, 226, 0.8)",
                    borderColor: "#4a90e2",
                    borderWidth: 1,
                    yAxisID: "y"
                }, {
                    label: "Hiệu suất (%)",
                    data: chartData.performances,
                    type: "line",
                    backgroundColor: "rgba(85, 239, 196, 0.2)",
                    borderColor: "#55efc4",
                    borderWidth: 3,
                    tension: 0.4,
                    yAxisID: "y1"
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: "bottom"
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                if (context.datasetIndex === 0) {
                                    return "Doanh thu: " + context.raw + " triệu đồng";
                                } else {
                                    return "Hiệu suất: " + context.raw + "%";
                                }
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        type: "linear",
                        display: true,
                        position: "left",
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: "Doanh thu (triệu đồng)"
                        }
                    },
                    y1: {
                        type: "linear",
                        display: true,
                        position: "right",
                        beginAtZero: true,
                        max: 100,
                        title: {
                            display: true,
                            text: "Hiệu suất (%)"
                        },
                        grid: {
                            drawOnChartArea: false,
                        }
                    }
                }
            }
        });
    }
}

// Branch actions
function viewBranch(id) {
    console.log("Viewing branch:", id);
    currentBranchId = id;Id = id;
    const modal = new bootstrap.Modal(document.getElementById("viewBranchModal"));
    
    fetch("/branches/viewDetail?id=" + id)
        .then(response => {
            if (!response.ok) {
                throw new Error("Network response was not ok");
            }
            return response.json();
        })
        .then(data => {
            console.log("Branch data:", data);
            if (data.success) {
                displayBranchDetails(data.branch, data.managers, data.performance);
            } else {
                showNotification(data.message || "Có lỗi xảy ra", "error");
            }
        })
        .catch(error => {
            console.error("Error:", error);
            showNotification("Có lỗi xảy ra khi tải thông tin chi nhánh", "error");
        });
    
    modal.show();
}

function editBranch(id) {
    console.log("Editing branch:", id);
    currentBranchId = id;
    const modal = new bootstrap.Modal(document.getElementById("editBranchModal"));
    
    fetch("/branches/viewDetail?id=" + id)
        .then(response => {
            if (!response.ok) {
                throw new Error("Network response was not ok");
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                displayEditForm(data.branch, data.managers);
            } else {
                showNotification(data.message || "Có lỗi xảy ra", "error");
            }
        })
        .catch(error => {
            console.error("Error:", error);
            showNotification("Có lỗi xảy ra khi tải thông tin chi nhánh", "error");
        });
    
    modal.show();
}

function deleteBranch(id) {
    console.log("Deleting branch:", id);
    currentBranchId = id;
    
    // Get branch name for confirmation
    fetch("/branches/viewDetail?id=" + id)
        .then(response => {
            if (!response.ok) {
                throw new Error("Network response was not ok");
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                document.getElementById("deleteBranchName").textContent = data.branch.name;
                const modal = new bootstrap.Modal(document.getElementById("deleteBranchModal"));
                modal.show();
            } else {
                showNotification(data.message || "Có lỗi xảy ra", "error");
            }
        })
        .catch(error => {
            console.error("Error:", error);
            showNotification("Có lỗi xảy ra khi tải thông tin chi nhánh", "error");
        });
}

function viewReport(id) {
    console.log("Viewing report for branch:", id);
    showNotification("Tính năng báo cáo sẽ được phát triển trong phiên bản tiếp theo", "info");
}

// Display functions
function displayBranchDetails(branch, managers, performance) {
    console.log("Displaying branch details:", branch);
    
    const statusText = branch.status == "active" ? "Hoạt động" : 
                      (branch.status == "maintenance" ? "Bảo trì" : "Tạm dừng");
    const statusColor = branch.status == "active" ? "success" : 
                       (branch.status == "maintenance" ? "warning" : "secondary");
    
    const content = `
        <div class="row">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">Thông tin chi nhánh</h6>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <p><strong>Tên:</strong> ${escapeHtml(branch.name)}</p>
                                <p><strong>Mã:</strong> ${escapeHtml(branch.code || "N/A")}</p>
                                <p><strong>Địa chỉ:</strong> ${escapeHtml(branch.address)}</p>
                                <p><strong>Điện thoại:</strong> ${escapeHtml(branch.phone)}</p>
                            </div>
                            <div class="col-md-6">
                                <p><strong>Email:</strong> ${escapeHtml(branch.email || "N/A")}</p>
                                <p><strong>Ngày khai trương:</strong> ${escapeHtml(branch.open_date || "N/A")}</p>
                                <p><strong>Trạng thái:</strong> <span class="badge bg-${statusColor}">${statusText}</span></p>
                                <p><strong>Mục tiêu doanh thu:</strong> ${formatCurrency(branch.target_revenue || 0)}</p>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="card mt-3">
                    <div class="card-header">
                        <h6 class="mb-0">Danh sách quản lý</h6>
                    </div>
                    <div class="card-body">
                        ${managers && managers.length > 0 ? managers.map(m => `
                            <div class="d-flex align-items-center mb-2">
                                <div class="me-3">
                                    <span class="badge bg-${m.role === "primary" ? "primary" : "secondary"}">${m.role}</span>
                                </div>
                                <div>
                                    <div class="fw-bold">${escapeHtml(m.name)}</div>
                                    <div class="text-muted small">${escapeHtml(m.email)}</div>
                                </div>
                            </div>
                        `).join("") : "<p class=\"text-muted\">Chưa có quản lý nào</p>"}
                    </div>
                </div>
            </div>
            
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">Hiệu suất tháng này</h6>
                    </div>
                    <div class="card-body">
                        <div class="text-center">
                            <div class="display-6 text-primary">${performance ? (performance.target_achievement || 0) : 0}%</div>
                            <p class="text-muted">Đạt mục tiêu</p>
                        </div>
                        <hr>
                        <p><strong>Doanh thu:</strong> ${formatCurrency(performance ? performance.total_revenue : 0)}</p>
                        <p><strong>Số giao dịch:</strong> ${performance ? performance.total_sales : 0}</p>
                        <p><strong>Nhân viên hoạt động:</strong> ${performance ? performance.active_employees : 0}</p>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    document.getElementById("viewBranchContent").innerHTML = content;
}

function displayEditForm(branch, managers) {
    console.log("Displaying edit form for branch:", branch);
    
    const content = `
        <input type="hidden" name="branch_id" value="${branch.id}">
        <div class="row">
            <div class="col-md-6">
                <div class="mb-3">
                    <label for="editBranchName" class="form-label">Tên chi nhánh</label>
                    <input type="text" class="form-control" id="editBranchName" name="name" value="${escapeHtml(branch.name)}" required>
                </div>
            </div>
            <div class="col-md-6">
                <div class="mb-3">
                    <label for="editBranchCode" class="form-label">Mã chi nhánh</label>
                    <input type="text" class="form-control" id="editBranchCode" name="code" value="${escapeHtml(branch.code || "")}" required>
                </div>
            </div>
        </div>
        <div class="mb-3">
            <label for="editBranchAddress" class="form-label">Địa chỉ</label>
            <textarea class="form-control" id="editBranchAddress" name="address" rows="2" required>${escapeHtml(branch.address)}</textarea>
        </div>
        <div class="row">
            <div class="col-md-6">
                <div class="mb-3">
                    <label for="editBranchPhone" class="form-label">Số điện thoại</label>
                    <input type="tel" class="form-control" id="editBranchPhone" name="phone" value="${escapeHtml(branch.phone)}" required>
                </div>
            </div>
            <div class="col-md-6">
                <div class="mb-3">
                    <label for="editBranchEmail" class="form-label">Email</label>
                    <input type="email" class="form-control" id="editBranchEmail" name="email" value="${escapeHtml(branch.email || "")}">
                </div>
            </div>
        </div>
        <div class="row">
            <div class="col-md-6">
                <div class="mb-3">
                    <label for="editBranchManagers" class="form-label">Quản lý chi nhánh</label>
                    <select class="form-select" id="editBranchManagers" name="managers[]" multiple size="4">
                        <option value="">Chọn quản lý (có thể chọn nhiều)</option>
                        ' . getManagersOptions($availableManagers, $managers ?? []) . '
                    </select>
                    <small class="text-muted">Giữ Ctrl để chọn nhiều quản lý. Quản lý đầu tiên sẽ là quản lý chính.</small>
                </div>
            </div>
            <div class="col-md-6">
                <div class="mb-3">
                    <label for="editBranchOpenDate" class="form-label">Ngày khai trương</label>
                    <input type="date" class="form-control" id="editBranchOpenDate" name="open_date" value="${branch.open_date}" required>
                </div>
                <div class="mb-3">
                    <label for="editTargetRevenue" class="form-label">Mục tiêu doanh thu (VND)</label>
                    <input type="number" class="form-control" id="editTargetRevenue" name="target_revenue" value="${branch.target_revenue || 50000000}" min="0">
                </div>
            </div>
        </div>
    `;
    
    document.getElementById("editBranchContent").innerHTML = content;
}

// Utility functions
function escapeHtml(unsafe) {
    if (unsafe === null || unsafe === undefined) return "";
    return unsafe
        .toString()
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/\'/g, "&#039;");
}

function formatCurrency(amount) {
    return new Intl.NumberFormat("vi-VN").format(amount) + "đ";
}

function showNotification(message, type = "info") {
    console.log("Showing notification:", message, type);
    
    const toast = document.createElement("div");
    toast.className = `toast align-items-center text-white bg-${type} border-0`;
    toast.setAttribute("role", "alert");
    toast.innerHTML = `
        <div class="d-flex">
            <div class="toast-body">${escapeHtml(message)}</div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
        </div>
    `;
    
    let container = document.getElementById("toast-container");
    if (!container) {
        container = document.createElement("div");
        container.id = "toast-container";
        container.className = "toast-container position-fixed top-0 end-0 p-3";
        container.style.zIndex = "9999";
        document.body.appendChild(container);
    }
    
    container.appendChild(toast);
    
    const bsToast = new bootstrap.Toast(toast);
    bsToast.show();
    
    toast.addEventListener("hidden.bs.toast", () => {
        toast.remove();
    });
}

// Form submissions
document.addEventListener("DOMContentLoaded", function() {
    // Add branch form
    const addBranchForm = document.getElementById("addBranchForm");
    if (addBranchForm) {
        addBranchForm.addEventListener("submit", function(e) {
            e.preventDefault();
            console.log("Submitting add branch form");
            
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);
            
            // Handle multiple managers
            const managersSelect = document.getElementById("branchManagers");
            const managers = Array.from(managersSelect.selectedOptions).map(option => option.value);
            data.managers = managers;
            
            console.log("Sending data:", data);
            
            fetch("/api/branches/create", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                body: JSON.stringify(data)
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error("Network response was not ok");
                }
                return response.json();
            })
            .then(data => {
                console.log("Response:", data);
                if (data.success) {
                    const modal = bootstrap.Modal.getInstance(document.getElementById("addBranchModal"));
                    if (modal) modal.hide();
                    showNotification(data.message || "Chi nhánh đã được tạo thành công!", "success");
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(data.message || "Có lỗi xảy ra", "error");
                }
            })
            .catch(error => {
                console.error("Error:", error);
                showNotification("Có lỗi xảy ra. Vui lòng thử lại.", "error");
            });
        });
    }
    
    // Edit branch form
    const editBranchForm = document.getElementById("editBranchForm");
    if (editBranchForm) {
        editBranchForm.addEventListener("submit", function(e) {
            e.preventDefault();
            console.log("Submitting edit branch form");
            
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);
            
            // Handle multiple managers
            const managersSelect = document.getElementById("editBranchManagers");
            const managers = Array.from(managersSelect.selectedOptions).map(option => option.value);
            data.managers = managers;
            data.branch_id = currentBranchId;
            
            console.log("Sending data:", data);
            
            fetch("/api/branches/update", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                body: JSON.stringify(data)
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error("Network response was not ok");
                }
                return response.json();
            })
            .then(data => {
                console.log("Response:", data);
                if (data.success) {
                    const modal = bootstrap.Modal.getInstance(document.getElementById("editBranchModal"));
                    if (modal) modal.hide();
                    showNotification(data.message || "Chi nhánh đã được cập nhật thành công!", "success");
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(data.message || "Có lỗi xảy ra", "error");
                }
            })
            .catch(error => {
                console.error("Error:", error);
                showNotification("Có lỗi xảy ra. Vui lòng thử lại.", "error");
            });
        });
    }
    
    // Delete confirmation
    const confirmDeleteBtn = document.getElementById("confirmDeleteBranch");
    if (confirmDeleteBtn) {
        confirmDeleteBtn.addEventListener("click", function() {
            if (!currentBranchId) {
                showNotification("Không tìm thấy ID chi nhánh", "error");
                return;
            }
            
            console.log("Deleting branch:", currentBranchId);
            
            fetch("/api/branches/delete", {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                },
                body: JSON.stringify({ branch_id: currentBranchId })
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error("Network response was not ok");
                }
                return response.json();
            })
            .then(data => {
                console.log("Response:", data);
                if (data.success) {
                    const modal = bootstrap.Modal.getInstance(document.getElementById("deleteBranchModal"));
                    if (modal) modal.hide();
                    showNotification(data.message || "Chi nhánh đã được xóa thành công!", "success");
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showNotification(data.message || "Có lỗi xảy ra", "error");
                }
            })
            .catch(error => {
                console.error("Error:", error);
                showNotification("Có lỗi xảy ra. Vui lòng thử lại.", "error");
            });
        });
    }
    
    // Initialize chart
    initializeChart();
    
    // Set default open date
    const openDateInput = document.getElementById("branchOpenDate");
    if (openDateInput) {
        openDateInput.value = new Date().toISOString().split("T")[0];
    }
});

// Quick actions
function compareBranches() {
    fetch("/branches/comparison?period=month")
        .then(response => {
            if (!response.ok) {
                throw new Error("Network response was not ok");
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                updateChart(data.comparison);
                showNotification("Dữ liệu so sánh đã được cập nhật", "info");
            }
        })
        .catch(error => {
            console.error("Error:", error);
            showNotification("Có lỗi xảy ra khi tải dữ liệu so sánh", "error");
        });
}

function exportBranchReport() {
    showNotification("Tính năng xuất báo cáo sẽ được phát triển trong phiên bản tiếp theo", "info");
}

function scheduleMeeting() {
    showNotification("Tính năng lên lịch họp sẽ được phát triển trong phiên bản tiếp theo", "info");
}

function sendAlert() {
    showNotification("Tính năng gửi cảnh báo sẽ được phát triển trong phiên bản tiếp theo", "info");
}

// Chart period change
document.querySelectorAll("input[name=\"chartPeriod\"]").forEach(radio => {
    radio.addEventListener("change", function() {
        const period = this.id === "thisMonth" ? "month" : (this.id === "lastMonth" ? "last_month" : "quarter");
        updateChartPeriod(period);
    });
});

function updateChartPeriod(period) {
    fetch("/branches/comparison?period=" + period)
        .then(response => {
            if (!response.ok) {
                throw new Error("Network response was not ok");
            }
            return response.json();
        })
        .then(data => {
            if (data.success) {
                updateChart(data.comparison);
            }
        })
        .catch(error => {
            console.error("Error:", error);
            showNotification("Có lỗi xảy ra khi cập nhật biểu đồ", "error");
        });
}

function updateChart(data) {
    if (branchChart && data) {
        branchChart.data.labels = data.map(b => b.name);
        branchChart.data.datasets[0].data = data.map(b => Math.round(b.total_revenue / 1000000));
        branchChart.data.datasets[1].data = data.map(b => b.performance_score);
        branchChart.update();
    }
}
</script>';

// Helper function for manager options
function getManagersOptions($availableManagers, $currentManagers = []) {
    $options = '';
    if (!empty($availableManagers)) {
        foreach ($availableManagers as $manager) {
            $selected = '';
            if (!empty($currentManagers)) {
                foreach ($currentManagers as $current) {
                    if (isset($current['user_id']) && $current['user_id'] == $manager['id']) {
                        $selected = 'selected';
                        break;
                    }
                }
            }
            $options .= '<option value="' . $manager['id'] . '" ' . $selected . '>' . 
                       htmlspecialchars($manager['name']) . ' (' . ucfirst($manager['role']) . ')</option>';
        }
    }
    return $options;
}

include 'layout.php';
?>