<?php
class User {
    private $conn;
    private $table_name = "users";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function getUserByEmail($email) {
        $query = "SELECT * FROM users WHERE email = :email AND is_active = 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':email', $email);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getUserById($id) {
        $query = "SELECT u.*, b.name as branch_name, d.name as department_name
                  FROM users u
                  LEFT JOIN branches b ON u.branch_id = b.id
                  LEFT JOIN departments d ON u.department_id = d.id
                  WHERE u.id = :id AND u.is_active = 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function createUser($data) {
        $query = "INSERT INTO users (name, email, password, role, branch_id, department_id, phone, created_at) 
                  VALUES (:name, :email, :password, :role, :branch_id, :department_id, :phone, NOW())";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':name', $data['name']);
        $stmt->bindParam(':email', $data['email']);
        $stmt->bindParam(':password', password_hash($data['password'], PASSWORD_DEFAULT));
        $stmt->bindParam(':role', $data['role']);
        $stmt->bindParam(':branch_id', $data['branch_id']);
        $stmt->bindParam(':department_id', $data['department_id']);
        $stmt->bindParam(':phone', $data['phone']);
        
        return $stmt->execute();
    }

    public function updateUser($id, $data) {
        $query = "UPDATE users 
                  SET name = :name, email = :email, role = :role, 
                      branch_id = :branch_id, department_id = :department_id, 
                      phone = :phone, updated_at = NOW()
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->bindParam(':name', $data['name']);
        $stmt->bindParam(':email', $data['email']);
        $stmt->bindParam(':role', $data['role']);
        $stmt->bindParam(':branch_id', $data['branch_id']);
        $stmt->bindParam(':department_id', $data['department_id']);
        $stmt->bindParam(':phone', $data['phone']);
        
        return $stmt->execute();
    }

    public function updatePassword($id, $newPassword) {
        $query = "UPDATE users SET password = :password, updated_at = NOW() WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->bindParam(':password', password_hash($newPassword, PASSWORD_DEFAULT));
        
        return $stmt->execute();
    }

    public function deactivateUser($id) {
        $query = "UPDATE users SET is_active = 0, updated_at = NOW() WHERE id = :id AND role != 'owner'";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        
        return $stmt->execute();
    }

    public function getAllUsers() {
        $query = "SELECT u.*, b.name as branch_name, d.name as department_name
                  FROM users u
                  LEFT JOIN branches b ON u.branch_id = b.id
                  LEFT JOIN departments d ON u.department_id = d.id
                  WHERE u.is_active = 1
                  ORDER BY u.name ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getUsersByRole($role) {
        $query = "SELECT u.*, b.name as branch_name, d.name as department_name
                  FROM users u
                  LEFT JOIN branches b ON u.branch_id = b.id
                  LEFT JOIN departments d ON u.department_id = d.id
                  WHERE u.role = :role AND u.is_active = 1
                  ORDER BY u.name ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':role', $role);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getUsersByBranch($branchId) {
        $query = "SELECT u.*, b.name as branch_name, d.name as department_name
                  FROM users u
                  LEFT JOIN branches b ON u.branch_id = b.id
                  LEFT JOIN departments d ON u.department_id = d.id
                  WHERE u.branch_id = :branch_id AND u.is_active = 1
                  ORDER BY u.name ASC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branchId);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function searchUsers($searchTerm) {
        $query = "SELECT u.*, b.name as branch_name, d.name as department_name
                  FROM users u
                  LEFT JOIN branches b ON u.branch_id = b.id
                  LEFT JOIN departments d ON u.department_id = d.id
                  WHERE (u.name LIKE :search OR u.email LIKE :search) 
                  AND u.is_active = 1
                  ORDER BY u.name ASC";
        
        $stmt = $this->conn->prepare($query);
        $searchParam = '%' . $searchTerm . '%';
        $stmt->bindParam(':search', $searchParam);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getUserStats() {
        $query = "SELECT 
                    COUNT(*) as total_users,
                    SUM(CASE WHEN role = 'owner' THEN 1 ELSE 0 END) as owners,
                    SUM(CASE WHEN role = 'manager' THEN 1 ELSE 0 END) as managers,
                    SUM(CASE WHEN role = 'team_leader' THEN 1 ELSE 0 END) as team_leaders,
                    SUM(CASE WHEN role = 'employee' THEN 1 ELSE 0 END) as employees,
                    SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_users,
                    SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive_users
                  FROM users";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
?>