<?php
class Department {
    private $conn;
    private $table_name = "departments";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function getAllDepartments() {
        $query = "SELECT d.*, b.name as branch_name, u.name as leader_name,
                         COUNT(DISTINCT e.id) as employee_count
                  FROM departments d
                  LEFT JOIN branches b ON d.branch_id = b.id
                  LEFT JOIN users u ON d.leader_id = u.id
                  LEFT JOIN users e ON d.id = e.department_id AND e.is_active = 1
                  GROUP BY d.id
                  ORDER BY b.name, d.name";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getDepartmentsByBranch($branch_id) {
        $query = "SELECT d.*, b.name as branch_name, u.name as leader_name,
                         COUNT(DISTINCT e.id) as employee_count
                  FROM departments d
                  LEFT JOIN branches b ON d.branch_id = b.id
                  LEFT JOIN users u ON d.leader_id = u.id
                  LEFT JOIN users e ON d.id = e.department_id AND e.is_active = 1
                  WHERE d.branch_id = :branch_id
                  GROUP BY d.id
                  ORDER BY d.name";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getDepartmentById($id) {
        $query = "SELECT d.*, b.name as branch_name, u.name as leader_name
                  FROM departments d
                  LEFT JOIN branches b ON d.branch_id = b.id
                  LEFT JOIN users u ON d.leader_id = u.id
                  WHERE d.id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function createDepartment($data) {
        $query = "INSERT INTO departments (branch_id, name, description, leader_id, color, created_at) 
                  VALUES (:branch_id, :name, :description, :leader_id, :color, NOW())";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $data['branch_id']);
        $stmt->bindParam(':name', $data['name']);
        $stmt->bindParam(':description', $data['description']);
        $stmt->bindParam(':leader_id', $data['leader_id']);
        $stmt->bindParam(':color', $data['color']);
        
        return $stmt->execute();
    }

    public function updateDepartment($id, $data) {
        $query = "UPDATE departments 
                  SET name = :name, description = :description, 
                      leader_id = :leader_id, color = :color, updated_at = NOW()
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->bindParam(':name', $data['name']);
        $stmt->bindParam(':description', $data['description']);
        $stmt->bindParam(':leader_id', $data['leader_id']);
        $stmt->bindParam(':color', $data['color']);
        
        return $stmt->execute();
    }

    public function deleteDepartment($id) {
        // Kiểm tra xem có nhân viên nào trong phòng ban không
        $query = "SELECT COUNT(*) as count FROM users WHERE department_id = :id AND is_active = 1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result['count'] > 0) {
            return false; // Không thể xóa nếu còn nhân viên
        }
        
        $query = "DELETE FROM departments WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':id', $id);
        
        return $stmt->execute();
    }

    public function getDepartmentStats($department_id) {
        $query = "SELECT 
                    COUNT(DISTINCT u.id) as employee_count,
                    COALESCE(SUM(s.amount), 0) as monthly_revenue,
                    COUNT(DISTINCT t.id) as total_tasks,
                    COUNT(DISTINCT tc.task_id) as completed_tasks
                  FROM departments d
                  LEFT JOIN users u ON d.id = u.department_id AND u.is_active = 1
                  LEFT JOIN sales s ON u.id = s.user_id AND MONTH(s.sale_date) = MONTH(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())
                  LEFT JOIN daily_tasks t ON d.id = t.department_id
                  LEFT JOIN task_completions tc ON t.id = tc.task_id AND DATE(tc.completed_at) = CURDATE()
                  WHERE d.id = :department_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':department_id', $department_id);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
?>