<?php
class Branch {
    private $conn;
    private $table_name = "branches";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function getAllBranches() {
        $query = "SELECT b.*, 
                         GROUP_CONCAT(DISTINCT CONCAT(u.name, ' (', bm.role, ')') SEPARATOR ', ') as managers,
                         COUNT(DISTINCT e.id) as employee_count,
                         COALESCE(SUM(s.amount), 0) as monthly_revenue,
                         ROUND((COALESCE(SUM(s.amount), 0) / NULLIF(b.target_revenue, 0)) * 100, 2) as performance_score
                  FROM branches b
                  LEFT JOIN branch_managers bm ON b.id = bm.branch_id
                  LEFT JOIN users u ON bm.user_id = u.id
                  LEFT JOIN users e ON b.id = e.branch_id AND e.is_active = 1
                  LEFT JOIN sales s ON b.id = s.branch_id AND MONTH(s.sale_date) = MONTH(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())
                  WHERE b.status != 'deleted'
                  GROUP BY b.id
                  ORDER BY b.name";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getBranchById($branch_id) {
        $query = "SELECT b.*, 
                         GROUP_CONCAT(DISTINCT CONCAT(u.name, ' (', bm.role, ')') SEPARATOR ', ') as managers,
                         COUNT(DISTINCT e.id) as employee_count,
                         COALESCE(SUM(s.amount), 0) as monthly_revenue,
                         ROUND((COALESCE(SUM(s.amount), 0) / NULLIF(b.target_revenue, 0)) * 100, 2) as performance_score
                  FROM branches b
                  LEFT JOIN branch_managers bm ON b.id = bm.branch_id
                  LEFT JOIN users u ON bm.user_id = u.id
                  LEFT JOIN users e ON b.id = e.branch_id AND e.is_active = 1
                  LEFT JOIN sales s ON b.id = s.branch_id AND MONTH(s.sale_date) = MONTH(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())
                  WHERE b.id = :branch_id AND b.status != 'deleted'
                  GROUP BY b.id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function createBranch($data) {
        try {
            $this->conn->beginTransaction();
            
            // Insert branch
            $query = "INSERT INTO branches (name, code, address, phone, email, open_date, status, target_revenue, created_at) 
                      VALUES (:name, :code, :address, :phone, :email, :open_date, 'active', :target_revenue, NOW())";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':name', $data['name']);
            $stmt->bindParam(':code', $data['code']);
            $stmt->bindParam(':address', $data['address']);
            $stmt->bindParam(':phone', $data['phone']);
            $stmt->bindParam(':email', $data['email']);
            $stmt->bindParam(':open_date', $data['open_date']);
            $stmt->bindParam(':target_revenue', $data['target_revenue'] ?? 50000000);
            
            if (!$stmt->execute()) {
                throw new Exception('Failed to create branch');
            }
            
            $branchId = $this->conn->lastInsertId();
            
            // Insert managers
            if (!empty($data['managers']) && is_array($data['managers'])) {
                foreach ($data['managers'] as $index => $managerId) {
                    $role = $index === 0 ? 'primary' : 'secondary';
                    $this->addBranchManager($branchId, $managerId, $role);
                }
            }
            
            $this->conn->commit();
            return $branchId;
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }

    public function updateBranch($branch_id, $data) {
        try {
            $this->conn->beginTransaction();
            
            // Update branch
            $query = "UPDATE branches 
                      SET name = :name, code = :code, address = :address, phone = :phone, 
                          email = :email, open_date = :open_date, target_revenue = :target_revenue, updated_at = NOW()
                      WHERE id = :branch_id";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':branch_id', $branch_id);
            $stmt->bindParam(':name', $data['name']);
            $stmt->bindParam(':code', $data['code']);
            $stmt->bindParam(':address', $data['address']);
            $stmt->bindParam(':phone', $data['phone']);
            $stmt->bindParam(':email', $data['email']);
            $stmt->bindParam(':open_date', $data['open_date']);
            $stmt->bindParam(':target_revenue', $data['target_revenue']);
            
            if (!$stmt->execute()) {
                throw new Exception('Failed to update branch');
            }
            
            // Update managers
            if (isset($data['managers']) && is_array($data['managers'])) {
                // Remove existing managers
                $this->removeBranchManagers($branch_id);
                
                // Add new managers
                foreach ($data['managers'] as $index => $managerId) {
                    $role = $index === 0 ? 'primary' : 'secondary';
                    $this->addBranchManager($branch_id, $managerId, $role);
                }
            }
            
            $this->conn->commit();
            return true;
            
        } catch (Exception $e) {
            $this->conn->rollBack();
            return false;
        }
    }

    public function deleteBranch($branch_id) {
        $query = "UPDATE branches SET status = 'deleted', updated_at = NOW() WHERE id = :branch_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        return $stmt->execute();
    }

    public function addBranchManager($branch_id, $user_id, $role = 'secondary') {
        $query = "INSERT INTO branch_managers (branch_id, user_id, role, assigned_at) 
                  VALUES (:branch_id, :user_id, :role, NOW())
                  ON DUPLICATE KEY UPDATE role = :role, assigned_at = NOW()";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->bindParam(':role', $role);
        
        return $stmt->execute();
    }

    public function removeBranchManagers($branch_id) {
        $query = "DELETE FROM branch_managers WHERE branch_id = :branch_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        return $stmt->execute();
    }

    public function getBranchManagers($branch_id) {
        $query = "SELECT bm.*, u.name, u.email, u.role as user_role
                  FROM branch_managers bm
                  JOIN users u ON bm.user_id = u.id
                  WHERE bm.branch_id = :branch_id
                  ORDER BY FIELD(bm.role, 'primary', 'secondary', 'assistant'), bm.assigned_at";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getAvailableManagers() {
        $query = "SELECT id, name, email, role 
                  FROM users 
                  WHERE role IN ('manager', 'team_leader') AND is_active = 1
                  ORDER BY name";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getBranchPerformance($branch_id, $period = 'month') {
        $dateCondition = '';
        switch($period) {
            case 'week':
                $dateCondition = "AND WEEK(s.sale_date) = WEEK(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())";
                break;
            case 'quarter':
                $dateCondition = "AND QUARTER(s.sale_date) = QUARTER(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())";
                break;
            case 'year':
                $dateCondition = "AND YEAR(s.sale_date) = YEAR(CURDATE())";
                break;
            default:
                $dateCondition = "AND MONTH(s.sale_date) = MONTH(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())";
        }
        
        $query = "SELECT 
                    COUNT(DISTINCT s.id) as total_sales,
                    COALESCE(SUM(s.amount), 0) as total_revenue,
                    COALESCE(SUM(CASE WHEN s.type = 'service' THEN s.amount END), 0) as service_revenue,
                    COALESCE(SUM(CASE WHEN s.type = 'product' THEN s.amount END), 0) as product_revenue,
                    COUNT(DISTINCT s.user_id) as active_employees,
                    b.target_revenue,
                    ROUND((COALESCE(SUM(s.amount), 0) / NULLIF(b.target_revenue, 0)) * 100, 2) as target_achievement
                  FROM branches b
                  LEFT JOIN sales s ON b.id = s.branch_id $dateCondition
                  WHERE b.id = :branch_id
                  GROUP BY b.id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getBranchComparison($period = 'month') {
        $dateCondition = '';
        switch($period) {
            case 'week':
                $dateCondition = "AND WEEK(s.sale_date) = WEEK(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())";
                break;
            case 'quarter':
                $dateCondition = "AND QUARTER(s.sale_date) = QUARTER(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())";
                break;
            case 'year':
                $dateCondition = "AND YEAR(s.sale_date) = YEAR(CURDATE())";
                break;
            default:
                $dateCondition = "AND MONTH(s.sale_date) = MONTH(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())";
        }
        
        $query = "SELECT 
                    b.id, b.name, b.code,
                    COUNT(DISTINCT s.id) as total_sales,
                    COALESCE(SUM(s.amount), 0) as total_revenue,
                    COALESCE(SUM(CASE WHEN s.type = 'service' THEN s.amount END), 0) as service_revenue,
                    COALESCE(SUM(CASE WHEN s.type = 'product' THEN s.amount END), 0) as product_revenue,
                    COUNT(DISTINCT e.id) as employee_count,
                    b.target_revenue,
                    ROUND((COALESCE(SUM(s.amount), 0) / NULLIF(b.target_revenue, 0)) * 100, 2) as performance_score
                  FROM branches b
                  LEFT JOIN sales s ON b.id = s.branch_id $dateCondition
                  LEFT JOIN users e ON b.id = e.branch_id AND e.is_active = 1
                  WHERE b.status = 'active'
                  GROUP BY b.id
                  ORDER BY performance_score DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getBranchStats() {
        $query = "SELECT 
                    COUNT(*) as total_branches,
                    COUNT(CASE WHEN status = 'active' THEN 1 END) as active_branches,
                    COUNT(CASE WHEN status = 'inactive' THEN 1 END) as inactive_branches,
                    COUNT(CASE WHEN status = 'maintenance' THEN 1 END) as maintenance_branches,
                    COALESCE(SUM(target_revenue), 0) as total_target_revenue
                  FROM branches 
                  WHERE status != 'deleted'";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function updateBranchPerformance($branch_id, $period_type, $period_date, $data) {
        $query = "INSERT INTO branch_performance 
                    (branch_id, period_type, period_date, total_revenue, service_revenue, product_revenue, 
                     total_sales, active_employees, performance_score, target_achievement)
                  VALUES 
                    (:branch_id, :period_type, :period_date, :total_revenue, :service_revenue, :product_revenue,
                     :total_sales, :active_employees, :performance_score, :target_achievement)
                  ON DUPLICATE KEY UPDATE
                    total_revenue = :total_revenue,
                    service_revenue = :service_revenue,
                    product_revenue = :product_revenue,
                    total_sales = :total_sales,
                    active_employees = :active_employees,
                    performance_score = :performance_score,
                    target_achievement = :target_achievement,
                    updated_at = NOW()";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->bindParam(':period_type', $period_type);
        $stmt->bindParam(':period_date', $period_date);
        $stmt->bindParam(':total_revenue', $data['total_revenue']);
        $stmt->bindParam(':service_revenue', $data['service_revenue']);
        $stmt->bindParam(':product_revenue', $data['product_revenue']);
        $stmt->bindParam(':total_sales', $data['total_sales']);
        $stmt->bindParam(':active_employees', $data['active_employees']);
        $stmt->bindParam(':performance_score', $data['performance_score']);
        $stmt->bindParam(':target_achievement', $data['target_achievement']);
        
        return $stmt->execute();
    }
}
?>