<?php
require_once __DIR__ . '/../config/database.php';

class SettingsController extends Controller {
    protected $conn;
    public function index() {
        $this->conn = $this->db;
        
        // Kiểm tra quyền truy cập
        $permissionModel = $this->model('Permission');
        if(!$permissionModel->hasPermission($_SESSION['user_id'], 'view_system_settings')) {
            header('Location: /dashboard');
            exit();
        }
        
        $settingsModel = $this->model('Settings');
        $userModel = $this->model('User');
        
        // Lấy tất cả cài đặt
        $systemSettings = $settingsModel->getAllSettings();
        
        // Lấy danh sách users để quản lý quyền
        $users = [];
        if($permissionModel->hasPermission($_SESSION['user_id'], 'manage_user_permissions')) {
            $users = $userModel->getAllUsers();
        }
        
        // Lấy danh sách quyền
        $allPermissions = $permissionModel->getAllPermissions();
        
        // Lấy thông tin hệ thống
        $systemInfo = $this->getSystemInfo();
        
        $this->view('settings', [
            'settings' => $systemSettings,
            'users' => $users,
            'permissions' => $allPermissions,
            'systemInfo' => $systemInfo,
            'userPermissions' => $permissionModel->getUserPermissions($_SESSION['user_id'])
        ]);
    }
    
    public function updateSystemSettings() {
        header('Content-Type: application/json');
        
        $permissionModel = $this->model('Permission');
        if(!$permissionModel->hasPermission($_SESSION['user_id'], 'edit_system_settings')) {
            echo json_encode(['success' => false, 'message' => 'Không có quyền']);
            exit();
        }
        
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $settingsModel = $this->model('Settings');
            $success = true;
            
            foreach($input as $key => $value) {
                if(!$settingsModel->setSetting($key, $value)) {
                    $success = false;
                    break;
                }
            }
            
            // Ghi audit log
            $this->logAction('update_system_settings', 'system_settings', null, null, $input);
            
            echo json_encode(['success' => $success]);
        }
        exit();
    }
    
    public function updateUserPermissions() {
        header('Content-Type: application/json');
        
        $permissionModel = $this->model('Permission');
        if(!$permissionModel->hasPermission($_SESSION['user_id'], 'manage_user_permissions')) {
            echo json_encode(['success' => false, 'message' => 'Không có quyền']);
            exit();
        }
        
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $input = json_decode(file_get_contents('php://input'), true);
            
            $userId = $input['user_id'];
            $permissions = $input['permissions'];
            
            // Xóa tất cả quyền custom hiện tại
            $query = "DELETE FROM user_permissions WHERE user_id = :user_id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->execute();
            
            // Thêm quyền mới
            $success = true;
            foreach($permissions as $permission) {
                if(!$permissionModel->setUserPermission($userId, $permission, true)) {
                    $success = false;
                    break;
                }
            }
            
            // Ghi audit log
            $this->logAction('update_user_permissions', 'user_permissions', $userId, null, $permissions);
            
            echo json_encode(['success' => $success]);
        }
        exit();
    }
    
    public function backupSystem() {
        header('Content-Type: application/json');
        
        $permissionModel = $this->model('Permission');
        if(!$permissionModel->hasPermission($_SESSION['user_id'], 'backup_system')) {
            echo json_encode(['success' => false, 'message' => 'Không có quyền']);
            exit();
        }
        
        try {
            $backupFile = $this->createBackup();
            
            // Ghi audit log
            $this->logAction('create_backup', 'system_backups', null, null, ['filename' => $backupFile]);
            
            echo json_encode(['success' => true, 'filename' => $backupFile]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit();
    }
    
    public function clearCache() {
        header('Content-Type: application/json');
        
        $permissionModel = $this->model('Permission');
        if(!$permissionModel->hasPermission($_SESSION['user_id'], 'system_maintenance')) {
            echo json_encode(['success' => false, 'message' => 'Không có quyền']);
            exit();
        }
        
        try {
            // Xóa cache files (nếu có)
            $cacheDir = '../cache/';
            if (is_dir($cacheDir)) {
                $files = glob($cacheDir . '*');
                foreach($files as $file) {
                    if(is_file($file)) {
                        unlink($file);
                    }
                }
            }
            
            // Ghi audit log
            $this->logAction('clear_cache', 'system', null);
            
            echo json_encode(['success' => true]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit();
    }
    
    public function getSystemLogs() {
        header('Content-Type: application/json');
        
        $permissionModel = $this->model('Permission');
        if(!$permissionModel->hasPermission($_SESSION['user_id'], 'view_system_logs')) {
            echo json_encode(['success' => false, 'message' => 'Không có quyền']);
            exit();
        }
        
        $query = "SELECT al.*, u.name as user_name 
                  FROM audit_logs al 
                  LEFT JOIN users u ON al.user_id = u.id 
                  ORDER BY al.created_at DESC 
                  LIMIT 100";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode(['success' => true, 'logs' => $logs]);
        exit();
    }
    
    private function getSystemInfo() {
        return [
            'php_version' => phpversion(),
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
            'database_version' => $this->getDatabaseVersion(),
            'disk_space' => $this->getDiskSpace(),
            'memory_usage' => $this->getMemoryUsage(),
            'uptime' => $this->getSystemUptime()
        ];
    }
    
    private function getDatabaseVersion() {
        try {
            $stmt = $this->conn->query("SELECT VERSION() as version");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result['version'];
        } catch (Exception $e) {
            return 'Unknown';
        }
    }
    
    private function getDiskSpace() {
        $bytes = disk_free_space(".");
        $si_prefix = array( 'B', 'KB', 'MB', 'GB', 'TB', 'EB', 'ZB', 'YB' );
        $base = 1024;
        $class = min((int)log($bytes , $base) , count($si_prefix) - 1);
        return sprintf('%1.2f' , $bytes / pow($base,$class)) . ' ' . $si_prefix[$class];
    }
    
    private function getMemoryUsage() {
        return round(memory_get_usage(true) / 1024 / 1024, 2) . ' MB';
    }
    
    private function getSystemUptime() {
        // Giả lập uptime
        return '15 ngày 8 giờ';
    }
    
    private function createBackup() {
        $backupDir = '../backups/';
        if (!is_dir($backupDir)) {
            mkdir($backupDir, 0755, true);
        }
        
        $filename = 'backup_' . date('Y-m-d_H-i-s') . '.sql';
        $filepath = $backupDir . $filename;
        
        // Tạo backup database
        $command = "mysqldump --user=" . DB_USER . " --password=" . DB_PASS . " --host=" . DB_HOST . " " . DB_NAME . " > " . $filepath;
        exec($command, $output, $return_var);
        
        if ($return_var !== 0) {
            throw new Exception('Backup failed');
        }
        
        // Lưu thông tin backup vào database
        $query = "INSERT INTO system_backups (filename, file_path, file_size, backup_type, status, created_by, completed_at) 
                  VALUES (:filename, :filepath, :filesize, 'manual', 'completed', :user_id, NOW())";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':filename', $filename);
        $stmt->bindParam(':filepath', $filepath);
        $stmt->bindParam(':filesize', filesize($filepath));
        $stmt->bindParam(':user_id', $_SESSION['user_id']);
        $stmt->execute();
        
        return $filename;
    }
    
    private function logAction($action, $table = null, $recordId = null, $oldValues = null, $newValues = null) {
        $query = "INSERT INTO audit_logs (user_id, action, table_name, record_id, old_values, new_values, ip_address, user_agent) 
                  VALUES (:user_id, :action, :table_name, :record_id, :old_values, :new_values, :ip_address, :user_agent)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $_SESSION['user_id']);
        $stmt->bindParam(':action', $action);
        $stmt->bindParam(':table_name', $table);
        $stmt->bindParam(':record_id', $recordId);
        $stmt->bindParam(':old_values', $oldValues ? json_encode($oldValues) : null);
        $stmt->bindParam(':new_values', $newValues ? json_encode($newValues) : null);
        $stmt->bindParam(':ip_address', $_SERVER['REMOTE_ADDR'] ?? null);
        $stmt->bindParam(':user_agent', $_SERVER['HTTP_USER_AGENT'] ?? null);
        
        return $stmt->execute();
    }
}
?>