<?php
require_once __DIR__ . '/../config/database.php';

class BranchController extends Controller {
    protected $conn;
    
    public function __construct() {
        parent::__construct();
        $this->conn = $this->db;
    }
    
    protected function checkAuth() {
        if (!isset($_SESSION['user_id'])) {
            header('Location: /login');
            exit();
        }
    }

    protected function checkPermission($allowedRoles = []) {
        $this->checkAuth();
        
        if (!empty($allowedRoles) && !in_array($_SESSION['user_role'], $allowedRoles)) {
            http_response_code(403);
            die('Bạn không có quyền truy cập trang này');
        }
    }
    
    public function index() {
        $this->checkPermission(['owner', 'manager', 'team_leader']);
        
        $branchModel = $this->model('Branch');
        $userModel = $this->model('User');
        
        // Lấy danh sách chi nhánh
        if ($_SESSION['user_role'] == 'owner') {
            $branches = $branchModel->getAllBranches();
        } else {
            $branches = [$branchModel->getBranchById($_SESSION['branch_id'])];
        }
        
        // Lấy danh sách quản lý có thể chọn
        $availableManagers = $branchModel->getAvailableManagers();
        
        // Lấy thống kê tổng quan
        $branchStats = $branchModel->getBranchStats();
        
        // Lấy dữ liệu so sánh hiệu suất
        $performanceData = $branchModel->getBranchComparison('month');
        
        $this->view('branches', [
            'branches' => $branches,
            'availableManagers' => $availableManagers,
            'branchStats' => $branchStats,
            'performanceData' => $performanceData
        ]);
    }
    
    public function create() {
        header('Content-Type: application/json');
        $this->checkPermission(['owner']);
        
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $rawInput = file_get_contents('php://input');
            $input = json_decode($rawInput, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                echo json_encode(['success' => false, 'message' => 'Dữ liệu không hợp lệ: ' . json_last_error_msg()]);
                exit();
            }
            
            if (empty($input)) {
                echo json_encode(['success' => false, 'message' => 'Không có dữ liệu nhận được']);
                exit();
            }
            
            // Validate input
            $required = ['name', 'code', 'address', 'phone', 'open_date'];
            foreach ($required as $field) {
                if (empty($input[$field])) {
                    echo json_encode(['success' => false, 'message' => "Trường {$field} là bắt buộc"]);
                    exit();
                }
            }
            
            $branchModel = $this->model('Branch');
            
            // Chuẩn bị dữ liệu
            $data = [
                'name' => trim($input['name']),
                'code' => trim($input['code']),
                'address' => trim($input['address']),
                'phone' => trim($input['phone']),
                'email' => trim($input['email'] ?? ''),
                'open_date' => $input['open_date'],
                'target_revenue' => floatval($input['target_revenue'] ?? 50000000),
                'managers' => $input['managers'] ?? []
            ];
            
            $result = $branchModel->createBranch($data);
            
            if ($result) {
                // Ghi audit log
                $this->logAction('create_branch', 'branches', $result, null, $data);
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Chi nhánh đã được tạo thành công',
                    'branch_id' => $result
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Có lỗi xảy ra khi tạo chi nhánh. Có thể mã chi nhánh đã tồn tại.']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Phương thức không hợp lệ']);
        }
        exit();
    }
    
    public function update() {
        header('Content-Type: application/json');
        $this->checkPermission(['owner', 'manager']);
        
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $rawInput = file_get_contents('php://input');
            $input = json_decode($rawInput, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                echo json_encode(['success' => false, 'message' => 'Dữ liệu không hợp lệ: ' . json_last_error_msg()]);
                exit();
            }
            
            $branchId = intval($input['branch_id'] ?? 0);
            if (!$branchId) {
                echo json_encode(['success' => false, 'message' => 'ID chi nhánh không hợp lệ']);
                exit();
            }
            
            $branchModel = $this->model('Branch');
            
            // Kiểm tra quyền sửa chi nhánh
            if ($_SESSION['user_role'] == 'manager' && $_SESSION['branch_id'] != $branchId) {
                echo json_encode(['success' => false, 'message' => 'Bạn chỉ có thể sửa chi nhánh của mình']);
                exit();
            }
            
            $data = [
                'name' => trim($input['name']),
                'code' => trim($input['code']),
                'address' => trim($input['address']),
                'phone' => trim($input['phone']),
                'email' => trim($input['email'] ?? ''),
                'open_date' => $input['open_date'],
                'target_revenue' => floatval($input['target_revenue'] ?? 50000000),
                'managers' => $input['managers'] ?? []
            ];
            
            $result = $branchModel->updateBranch($branchId, $data);
            
            if ($result) {
                // Ghi audit log
                $this->logAction('update_branch', 'branches', $branchId, null, $data);
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Chi nhánh đã được cập nhật thành công'
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Có lỗi xảy ra khi cập nhật chi nhánh']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Phương thức không hợp lệ']);
        }
        exit();
    }
    
    public function delete() {
        header('Content-Type: application/json');
        $this->checkPermission(['owner']);
        
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $rawInput = file_get_contents('php://input');
            $input = json_decode($rawInput, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                echo json_encode(['success' => false, 'message' => 'Dữ liệu không hợp lệ: ' . json_last_error_msg()]);
                exit();
            }
            
            $branchId = intval($input['branch_id'] ?? 0);
            if (!$branchId) {
                echo json_encode(['success' => false, 'message' => 'ID chi nhánh không hợp lệ']);
                exit();
            }
            
            $branchModel = $this->model('Branch');
            $result = $branchModel->deleteBranch($branchId);
            
            if ($result) {
                // Ghi audit log
                $this->logAction('delete_branch', 'branches', $branchId);
                
                echo json_encode([
                    'success' => true, 
                    'message' => 'Chi nhánh đã được xóa thành công'
                ]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Có lỗi xảy ra khi xóa chi nhánh']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Phương thức không hợp lệ']);
        }
        exit();
    }
    
    public function viewDetail() {
        header('Content-Type: application/json');
        $this->checkAuth();
        
        $branchId = intval($_GET['id'] ?? 0);
        if (!$branchId) {
            echo json_encode(['success' => false, 'message' => 'ID chi nhánh không hợp lệ']);
            exit();
        }
        
        $branchModel = $this->model('Branch');
        
        // Kiểm tra quyền xem chi nhánh
        if ($_SESSION['user_role'] == 'manager' && $_SESSION['branch_id'] != $branchId) {
            echo json_encode(['success' => false, 'message' => 'Bạn chỉ có thể xem chi nhánh của mình']);
            exit();
        }
        
        $branch = $branchModel->getBranchById($branchId);
        if (!$branch) {
            echo json_encode(['success' => false, 'message' => 'Không tìm thấy chi nhánh']);
            exit();
        }
        
        $managers = $branchModel->getBranchManagers($branchId);
        $performance = $branchModel->getBranchPerformance($branchId, 'month');
        
        echo json_encode([
            'success' => true,
            'branch' => $branch,
            'managers' => $managers,
            'performance' => $performance
        ]);
        exit();
    }
    
    public function performance() {
        header('Content-Type: application/json');
        $this->checkAuth();
        
        $branchId = intval($_GET['branch_id'] ?? 0);
        $period = $_GET['period'] ?? 'month';
        
        if (!$branchId) {
            echo json_encode(['success' => false, 'message' => 'ID chi nhánh không hợp lệ']);
            exit();
        }
        
        $branchModel = $this->model('Branch');
        
        // Kiểm tra quyền xem hiệu suất
        if ($_SESSION['user_role'] == 'manager' && $_SESSION['branch_id'] != $branchId) {
            echo json_encode(['success' => false, 'message' => 'Bạn chỉ có thể xem hiệu suất chi nhánh của mình']);
            exit();
        }
        
        $performance = $branchModel->getBranchPerformance($branchId, $period);
        
        echo json_encode([
            'success' => true,
            'performance' => $performance
        ]);
        exit();
    }
    
    public function comparison() {
        header('Content-Type: application/json');
        $this->checkPermission(['owner', 'manager', 'team_leader']);
        
        $period = $_GET['period'] ?? 'month';
        
        $branchModel = $this->model('Branch');
        $comparison = $branchModel->getBranchComparison($period);
        
        echo json_encode([
            'success' => true,
            'comparison' => $comparison
        ]);
        exit();
    }
    
    public function getManagers() {
        header('Content-Type: application/json');
        $this->checkPermission(['owner', 'manager']);
        
        $branchModel = $this->model('Branch');
        $managers = $branchModel->getAvailableManagers();
        
        echo json_encode([
            'success' => true,
            'managers' => $managers
        ]);
        exit();
    }
    
    private function logAction($action, $table = null, $recordId = null, $oldValues = null, $newValues = null) {
        try {
            $query = "INSERT INTO audit_logs (user_id, action, table_name, record_id, old_values, new_values, ip_address, user_agent) 
                      VALUES (:user_id, :action, :table_name, :record_id, :old_values, :new_values, :ip_address, :user_agent)";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':user_id', $_SESSION['user_id']);
            $stmt->bindParam(':action', $action);
            $stmt->bindParam(':table_name', $table);
            $stmt->bindParam(':record_id', $recordId);
            $stmt->bindParam(':old_values', $oldValues ? json_encode($oldValues) : null);
            $stmt->bindParam(':new_values', $newValues ? json_encode($newValues) : null);
            $stmt->bindParam(':ip_address', $_SERVER['REMOTE_ADDR'] ?? null);
            $stmt->bindParam(':user_agent', $_SERVER['HTTP_USER_AGENT'] ?? null);
            
            return $stmt->execute();
        } catch (Exception $e) {
            error_log("Audit log error: " . $e->getMessage());
            return false;
        }
    }
}
?>