// Spa KPI Management System - Main Application JavaScript
class SpaKPIApp {
    constructor() {
        this.init();
        this.setupEventListeners();
        this.startRealtimeUpdates();
    }

    init() {
        console.log('Spa KPI System initialized');
        this.initializeCharts();
        this.initializeModals();
        this.initializeTooltips();
        this.setupSidebar();
    }

    setupEventListeners() {
        // Task management
        this.setupTaskHandlers();
        
        // Form submissions
        this.setupFormHandlers();
        
        // Real-time updates
        this.setupRealtimeHandlers();
        
        // Navigation
        this.setupNavigationHandlers();
    }

    setupTaskHandlers() {
        document.querySelectorAll('.task-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', (e) => {
                this.handleTaskToggle(e.target);
            });
        });

        document.querySelectorAll('.complete-task-btn').forEach(button => {
            button.addEventListener('click', (e) => {
                const taskId = e.target.dataset.taskId;
                const checkbox = document.querySelector(`#task${taskId}`);
                if (checkbox) {
                    checkbox.checked = true;
                    this.handleTaskToggle(checkbox);
                }
            });
        });
    }

    async handleTaskToggle(checkbox) {
        const taskId = checkbox.dataset.taskId;
        const completed = checkbox.checked;
        const taskItem = checkbox.closest('.task-item');

        try {
            // Optimistic UI update
            if (completed) {
                taskItem.classList.add('completed');
                taskItem.dataset.status = 'completed';
            } else {
                taskItem.classList.remove('completed');
                taskItem.dataset.status = 'pending';
            }

            const response = await fetch('/api/task/complete', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    task_id: taskId,
                    completed: completed
                })
            });

            const data = await response.json();

            if (data.success) {
                this.updateTaskStats();
                this.showNotification('Cập nhật nhiệm vụ thành công!', 'success');
                
                // Reload page after a short delay to show updated completion time
                if (completed) {
                    setTimeout(() => location.reload(), 1000);
                }
            } else {
                // Revert optimistic update
                checkbox.checked = !completed;
                if (completed) {
                    taskItem.classList.remove('completed');
                    taskItem.dataset.status = 'pending';
                } else {
                    taskItem.classList.add('completed');
                    taskItem.dataset.status = 'completed';
                }
                this.showNotification('Có lỗi xảy ra. Vui lòng thử lại.', 'error');
            }
        } catch (error) {
            console.error('Error:', error);
            // Revert optimistic update
            checkbox.checked = !completed;
            if (completed) {
                taskItem.classList.remove('completed');
                taskItem.dataset.status = 'pending';
            } else {
                taskItem.classList.add('completed');
                taskItem.dataset.status = 'completed';
            }
            this.showNotification('Có lỗi xảy ra. Vui lòng thử lại.', 'error');
        }
    }

    updateTaskStats() {
        const allTasks = document.querySelectorAll('.task-item');
        const completedTasks = document.querySelectorAll('.task-item.completed');
        const pendingTasks = allTasks.length - completedTasks.length;
        const completionRate = allTasks.length > 0 ? Math.round((completedTasks.length / allTasks.length) * 100) : 0;

        // Update stats if elements exist
        const totalTasksEl = document.getElementById('totalTasks');
        const completedTasksEl = document.getElementById('completedTasks');
        const pendingTasksEl = document.getElementById('pendingTasks');
        const completionRateEl = document.getElementById('completionRate');

        if (totalTasksEl) totalTasksEl.textContent = allTasks.length;
        if (completedTasksEl) completedTasksEl.textContent = completedTasks.length;
        if (pendingTasksEl) pendingTasksEl.textContent = pendingTasks;
        if (completionRateEl) completionRateEl.textContent = completionRate + '%';
    }

    setupFormHandlers() {
        // Handle all forms with ajax-form class
        document.querySelectorAll('.ajax-form').forEach(form => {
            form.addEventListener('submit', (e) => {
                e.preventDefault();
                this.submitForm(form);
            });
        });

        // Handle modal forms
        document.querySelectorAll('form[id$="Form"]').forEach(form => {
            form.addEventListener('submit', (e) => {
                e.preventDefault();
                this.submitModalForm(form);
            });
        });
    }

    async submitForm(form) {
        const submitBtn = form.querySelector('button[type="submit"]');
        const originalText = submitBtn.textContent;
        
        try {
            submitBtn.disabled = true;
            submitBtn.textContent = 'Đang xử lý...';

            const formData = new FormData(form);
            const response = await fetch(form.action, {
                method: form.method,
                body: formData
            });

            const data = await response.json();

            if (data.success) {
                this.showNotification('Thành công!', 'success');
                if (data.redirect) {
                    setTimeout(() => {
                        window.location.href = data.redirect;
                    }, 1000);
                } else {
                    form.reset();
                }
            } else {
                this.showNotification(data.message || 'Có lỗi xảy ra!', 'error');
            }
        } catch (error) {
            console.error('Error:', error);
            this.showNotification('Có lỗi xảy ra!', 'error');
        } finally {
            submitBtn.disabled = false;
            submitBtn.textContent = originalText;
        }
    }

    async submitModalForm(form) {
        const modal = form.closest('.modal');
        const submitBtn = form.querySelector('button[type="submit"]');
        const originalText = submitBtn.textContent;
        
        try {
            submitBtn.disabled = true;
            submitBtn.textContent = 'Đang xử lý...';

            const formData = new FormData(form);
            const data = Object.fromEntries(formData);

            const response = await fetch(form.action, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            });

            const result = await response.json();

            if (result.success) {
                // Close modal
                const modalInstance = bootstrap.Modal.getInstance(modal);
                modalInstance.hide();
                
                this.showNotification('Thành công!', 'success');
                
                // Reload page after short delay
                setTimeout(() => location.reload(), 500);
            } else {
                this.showNotification(result.message || 'Có lỗi xảy ra!', 'error');
            }
        } catch (error) {
            console.error('Error:', error);
            this.showNotification('Có lỗi xảy ra!', 'error');
        } finally {
            submitBtn.disabled = false;
            submitBtn.textContent = originalText;
        }
    }

    setupRealtimeHandlers() {
        // Auto-refresh data every 30 seconds for most pages
        if (!document.querySelector('.modal.show')) {
            setInterval(() => {
                this.refreshData();
            }, 30000);
        }

        // Refresh sales data more frequently
        if (window.location.pathname.includes('sales')) {
            setInterval(() => {
                this.refreshSalesData();
            }, 15000);
        }
    }

    async refreshData() {
        // Only refresh if no modals are open
        if (document.querySelector('.modal.show')) {
            return;
        }

        try {
            // Refresh current page data based on URL
            const path = window.location.pathname;
            
            if (path.includes('dashboard')) {
                // Refresh dashboard stats
                this.refreshDashboardStats();
            } else if (path.includes('tasks')) {
                // Refresh task list
                this.refreshTaskList();
            } else if (path.includes('kpi')) {
                // Refresh KPI data
                this.refreshKPIData();
            }
        } catch (error) {
            console.error('Error refreshing data:', error);
        }
    }

    async refreshSalesData() {
        try {
            const response = await fetch('/api/sales/recent');
            const data = await response.json();
            
            if (data.success) {
                this.updateRecentSales(data.sales);
            }
        } catch (error) {
            console.error('Error refreshing sales data:', error);
        }
    }

    updateRecentSales(sales) {
        const salesList = document.getElementById('recentSalesList');
        if (!salesList) return;

        // Update the sales list with new data
        // This would need to be implemented based on the specific HTML structure
    }

    setupNavigationHandlers() {
        // Handle sidebar navigation
        document.querySelectorAll('.sidebar .nav-link').forEach(link => {
            link.addEventListener('click', (e) => {
                // Remove active class from all links
                document.querySelectorAll('.sidebar .nav-link').forEach(l => {
                    l.classList.remove('active');
                });
                
                // Add active class to clicked link
                e.target.classList.add('active');
            });
        });

        // Handle mobile sidebar toggle
        const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
        if (mobileMenuBtn) {
            mobileMenuBtn.addEventListener('click', () => {
                this.toggleSidebar();
            });
        }

        // Handle sidebar overlay
        const sidebarOverlay = document.querySelector('.sidebar-overlay');
        if (sidebarOverlay) {
            sidebarOverlay.addEventListener('click', () => {
                this.toggleSidebar();
            });
        }
    }

    toggleSidebar() {
        const sidebar = document.querySelector('.sidebar');
        const overlay = document.querySelector('.sidebar-overlay');
        
        sidebar.classList.toggle('mobile-open');
        overlay.style.display = overlay.style.display === 'block' ? 'none' : 'block';
    }

    setupSidebar() {
        // Close sidebar when clicking on a link (mobile)
        document.querySelectorAll('.sidebar .nav-link').forEach(link => {
            link.addEventListener('click', () => {
                if (window.innerWidth < 768) {
                    this.toggleSidebar();
                }
            });
        });

        // Handle window resize
        window.addEventListener('resize', () => {
            if (window.innerWidth >= 768) {
                const sidebar = document.querySelector('.sidebar');
                const overlay = document.querySelector('.sidebar-overlay');
                
                sidebar.classList.remove('mobile-open');
                overlay.style.display = 'none';
            }
        });
    }

    initializeCharts() {
        // Initialize Chart.js charts if Chart is available
        if (typeof Chart !== 'undefined') {
            Chart.defaults.font.family = "'Segoe UI', Tahoma, Geneva, Verdana, sans-serif";
            Chart.defaults.color = '#6c757d';
            Chart.defaults.borderColor = 'rgba(0,0,0,0.1)';
        }
    }

    initializeModals() {
        // Set default datetime for modal forms
        document.querySelectorAll('input[type="datetime-local"]').forEach(input => {
            if (!input.value) {
                input.value = new Date().toISOString().slice(0, 16);
            }
        });

        // Set default date for date inputs
        document.querySelectorAll('input[type="date"]').forEach(input => {
            if (!input.value) {
                input.value = new Date().toISOString().split('T')[0];
            }
        });
    }

    initializeTooltips() {
        // Initialize Bootstrap tooltips
        if (typeof bootstrap !== 'undefined') {
            const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
        }
    }

    startRealtimeUpdates() {
        // Start real-time clock
        this.updateClock();
        setInterval(() => this.updateClock(), 1000);

        // Start periodic data refresh
        this.setupRealtimeHandlers();
    }

    updateClock() {
        const clockElements = document.querySelectorAll('.current-time');
        const now = new Date();
        const timeString = now.toLocaleTimeString('vi-VN');
        
        clockElements.forEach(el => {
            el.textContent = timeString;
        });
    }

    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
        notification.style.cssText = `
            top: 20px;
            right: 20px;
            z-index: 9999;
            min-width: 300px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        `;
        
        notification.innerHTML = `
            <div class="d-flex align-items-center">
                <i class="fas ${this.getNotificationIcon(type)} me-2"></i>
                <span>${message}</span>
                <button type="button" class="btn-close ms-auto" data-bs-dismiss="alert"></button>
            </div>
        `;

        document.body.appendChild(notification);

        // Auto remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 5000);
    }

    getNotificationIcon(type) {
        const icons = {
            success: 'fa-check-circle',
            error: 'fa-exclamation-circle',
            warning: 'fa-exclamation-triangle',
            info: 'fa-info-circle'
        };
        return icons[type] || icons.info;
    }

    // Utility methods
    formatCurrency(amount) {
        return new Intl.NumberFormat('vi-VN', {
            style: 'currency',
            currency: 'VND',
            minimumFractionDigits: 0
        }).format(amount);
    }

    formatNumber(number) {
        return new Intl.NumberFormat('vi-VN').format(number);
    }

    formatDate(date) {
        return new Intl.DateTimeFormat('vi-VN').format(new Date(date));
    }

    formatDateTime(date) {
        return new Intl.DateTimeFormat('vi-VN', {
            year: 'numeric',
            month: '2-digit',
            day: '2-digit',
            hour: '2-digit',
            minute: '2-digit'
        }).format(new Date(date));
    }
}

// Initialize the application when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    window.spaKPIApp = new SpaKPIApp();
});

// Global utility functions for backward compatibility
function showAlert(message, type) {
    if (window.spaKPIApp) {
        window.spaKPIApp.showNotification(message, type);
    }
}

function toggleSidebar() {
    if (window.spaKPIApp) {
        window.spaKPIApp.toggleSidebar();
    }
}