<?php
class Branch {
    private $conn;
    private $table_name = "branches";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function getAllBranches() {
        $query = "SELECT b.*, u.name as manager_name, u.email as manager_email,
                         COUNT(DISTINCT e.id) as employee_count,
                         COALESCE(SUM(s.amount), 0) as monthly_revenue
                  FROM branches b
                  LEFT JOIN users u ON b.manager_id = u.id
                  LEFT JOIN users e ON b.id = e.branch_id AND e.is_active = 1
                  LEFT JOIN sales s ON b.id = s.branch_id AND MONTH(s.sale_date) = MONTH(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())
                  GROUP BY b.id
                  ORDER BY b.name";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getBranchById($branch_id) {
        $query = "SELECT b.*, u.name as manager_name, u.email as manager_email,
                         COUNT(DISTINCT e.id) as employee_count,
                         COALESCE(SUM(s.amount), 0) as monthly_revenue
                  FROM branches b
                  LEFT JOIN users u ON b.manager_id = u.id
                  LEFT JOIN users e ON b.id = e.branch_id AND e.is_active = 1
                  LEFT JOIN sales s ON b.id = s.branch_id AND MONTH(s.sale_date) = MONTH(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())
                  WHERE b.id = :branch_id
                  GROUP BY b.id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function createBranch($name, $code, $address, $phone, $email, $manager_id) {
        $query = "INSERT INTO branches (name, address, phone, manager_id, created_at) 
                  VALUES (:name, :address, :phone, :manager_id, NOW())";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':address', $address);
        $stmt->bindParam(':phone', $phone);
        $stmt->bindParam(':manager_id', $manager_id);
        
        return $stmt->execute();
    }

    public function updateBranch($branch_id, $name, $address, $phone, $email, $manager_id) {
        $query = "UPDATE branches 
                  SET name = :name, address = :address, phone = :phone, 
                      email = :email, manager_id = :manager_id, updated_at = NOW()
                  WHERE id = :branch_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':address', $address);
        $stmt->bindParam(':phone', $phone);
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':manager_id', $manager_id);
        
        return $stmt->execute();
    }

    public function getBranchPerformance($branch_id, $period = 'month') {
        $dateCondition = '';
        switch($period) {
            case 'week':
                $dateCondition = "AND WEEK(s.sale_date) = WEEK(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())";
                break;
            case 'quarter':
                $dateCondition = "AND QUARTER(s.sale_date) = QUARTER(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())";
                break;
            default:
                $dateCondition = "AND MONTH(s.sale_date) = MONTH(CURDATE()) AND YEAR(s.sale_date) = YEAR(CURDATE())";
        }
        
        $query = "SELECT 
                    COUNT(DISTINCT s.id) as total_sales,
                    COALESCE(SUM(s.amount), 0) as total_revenue,
                    COALESCE(SUM(CASE WHEN s.type = 'service' THEN s.amount END), 0) as service_revenue,
                    COALESCE(SUM(CASE WHEN s.type = 'product' THEN s.amount END), 0) as product_revenue,
                    COUNT(DISTINCT s.user_id) as active_employees
                  FROM sales s
                  WHERE s.branch_id = :branch_id $dateCondition";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
?>