<?php
class Task {
    private $conn;
    private $table_name = "daily_tasks";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function getUserTasks($user_id, $branch_id) {
        $query = "SELECT t.id, t.title, t.description, 
                         CASE WHEN tc.id IS NOT NULL THEN 1 ELSE 0 END as completed,
                         tc.completed_at,
                         u.name as created_by
                  FROM daily_tasks t
                  LEFT JOIN task_completions tc ON t.id = tc.task_id AND tc.user_id = :user_id AND DATE(tc.completed_at) = CURDATE()
                  LEFT JOIN users u ON t.created_by = u.id
                  WHERE (t.branch_id = :branch_id OR t.branch_id IS NULL)
                  AND (t.end_date IS NULL OR t.end_date >= CURDATE())
                  ORDER BY t.id DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function createTask($title, $description, $branch_id, $user_id) {
        $query = "INSERT INTO daily_tasks (title, description, branch_id, created_by, created_at) 
                  VALUES (:title, :description, :branch_id, :user_id, NOW())";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':title', $title);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':branch_id', $branch_id);
        $stmt->bindParam(':user_id', $user_id);
        
        return $stmt->execute();
    }

    public function completeTask($task_id, $user_id) {
        $query = "INSERT INTO task_completions (task_id, user_id, completed_at) 
                  VALUES (:task_id, :user_id, NOW())
                  ON DUPLICATE KEY UPDATE completed_at = NOW()";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':task_id', $task_id);
        $stmt->bindParam(':user_id', $user_id);
        
        return $stmt->execute();
    }
    
    public function uncompleteTask($task_id, $user_id) {
        $query = "DELETE FROM task_completions 
                  WHERE task_id = :task_id AND user_id = :user_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':task_id', $task_id);
        $stmt->bindParam(':user_id', $user_id);
        
        return $stmt->execute();
    }
    
    public function deleteTask($task_id) {
        // Xóa các completion trước
        $query = "DELETE FROM task_completions WHERE task_id = :task_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':task_id', $task_id);
        $stmt->execute();
        
        // Xóa task
        $query = "DELETE FROM daily_tasks WHERE id = :task_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':task_id', $task_id);
        
        return $stmt->execute();
    }
}
?>