<?php
class Settings {
    private $conn;
    private $table_name = "system_settings";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function getSetting($key, $default = null) {
        $query = "SELECT value FROM system_settings WHERE setting_key = :key";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':key', $key);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result ? $result['value'] : $default;
    }
    
    public function setSetting($key, $value, $description = '') {
        $query = "INSERT INTO system_settings (setting_key, value, description, updated_at) 
                  VALUES (:key, :value, :description, NOW())
                  ON DUPLICATE KEY UPDATE 
                  value = :value, description = :description, updated_at = NOW()";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':key', $key);
        $stmt->bindParam(':value', $value);
        $stmt->bindParam(':description', $description);
        
        return $stmt->execute();
    }
    
    public function getAllSettings() {
        $query = "SELECT * FROM system_settings ORDER BY setting_key";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getSettingsByCategory($category) {
        $query = "SELECT * FROM system_settings WHERE setting_key LIKE :category ORDER BY setting_key";
        $stmt = $this->conn->prepare($query);
        $categoryPattern = $category . '%';
        $stmt->bindParam(':category', $categoryPattern);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function deleteSetting($key) {
        $query = "DELETE FROM system_settings WHERE setting_key = :key";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':key', $key);
        
        return $stmt->execute();
    }
    
    // Các phương thức helper cho các cài đặt thường dùng
    public function getSystemName() {
        return $this->getSetting('system_name', 'Spa KPI Management System');
    }
    
    public function getSystemVersion() {
        return $this->getSetting('system_version', '1.0.0');
    }
    
    public function getTimezone() {
        return $this->getSetting('timezone', 'Asia/Ho_Chi_Minh');
    }
    
    public function getCurrency() {
        return $this->getSetting('currency', 'VND');
    }
    
    public function getWorkingHours() {
        return [
            'start' => $this->getSetting('working_hours_start', '08:00'),
            'end' => $this->getSetting('working_hours_end', '22:00')
        ];
    }
    
    public function getAutoRefresh() {
        return $this->getSetting('auto_refresh', '1') == '1';
    }
    
    public function getNotifications() {
        return $this->getSetting('notifications_enabled', '1') == '1';
    }
    
    // KPI Settings
    public function getDefaultServiceTarget() {
        return (float)$this->getSetting('default_service_target', 50000000);
    }
    
    public function getDefaultProductTarget() {
        return (float)$this->getSetting('default_product_target', 20000000);
    }
    
    public function getKpiUpdateFrequency() {
        return $this->getSetting('kpi_update_frequency', 'realtime');
    }
    
    public function getPerformanceThreshold() {
        return (int)$this->getSetting('performance_threshold', 80);
    }
    
    // Security Settings
    public function getSessionTimeout() {
        return (int)$this->getSetting('session_timeout', 60);
    }
    
    public function getMaxLoginAttempts() {
        return (int)$this->getSetting('max_login_attempts', 5);
    }
    
    public function getTwoFactorAuth() {
        return $this->getSetting('two_factor_auth', '0') == '1';
    }
    
    public function getAuditLog() {
        return $this->getSetting('audit_log', '1') == '1';
    }
    
    // Email Settings
    public function getEmailSettings() {
        return [
            'smtp_host' => $this->getSetting('smtp_host', ''),
            'smtp_port' => $this->getSetting('smtp_port', '587'),
            'smtp_username' => $this->getSetting('smtp_username', ''),
            'smtp_password' => $this->getSetting('smtp_password', ''),
            'smtp_encryption' => $this->getSetting('smtp_encryption', 'tls'),
            'from_email' => $this->getSetting('from_email', ''),
            'from_name' => $this->getSetting('from_name', 'Spa KPI System')
        ];
    }
    
    // Backup Settings
    public function getBackupSettings() {
        return [
            'auto_backup' => $this->getSetting('auto_backup', '1') == '1',
            'backup_frequency' => $this->getSetting('backup_frequency', 'daily'),
            'backup_retention' => (int)$this->getSetting('backup_retention', 30),
            'backup_location' => $this->getSetting('backup_location', 'local')
        ];
    }
}
?>