<?php
class Permission {
    private $conn;
    
    // Định nghĩa các quyền trong hệ thống
    const PERMISSIONS = [
        // Dashboard & Reports
        'view_dashboard' => 'Xem dashboard',
        'view_reports' => 'Xem báo cáo',
        'export_reports' => 'Xuất báo cáo',
        
        // Tasks Management
        'view_tasks' => 'Xem nhiệm vụ',
        'create_tasks' => 'Tạo nhiệm vụ',
        'edit_tasks' => 'Sửa nhiệm vụ',
        'delete_tasks' => 'Xóa nhiệm vụ',
        'complete_tasks' => 'Hoàn thành nhiệm vụ',
        
        // KPI Management
        'view_kpi' => 'Xem KPI',
        'edit_kpi' => 'Chỉnh sửa KPI',
        'view_team_kpi' => 'Xem KPI team',
        'set_kpi_targets' => 'Đặt mục tiêu KPI',
        
        // Sales Management
        'view_sales' => 'Xem doanh số',
        'create_sales' => 'Thêm doanh số',
        'edit_sales' => 'Sửa doanh số',
        'delete_sales' => 'Xóa doanh số',
        'view_all_sales' => 'Xem tất cả doanh số',
        
        // Employee Management
        'view_employees' => 'Xem nhân viên',
        'create_employees' => 'Thêm nhân viên',
        'edit_employees' => 'Sửa nhân viên',
        'delete_employees' => 'Xóa nhân viên',
        'manage_employee_roles' => 'Quản lý chức vụ',
        
        // Branch Management
        'view_branches' => 'Xem chi nhánh',
        'create_branches' => 'Tạo chi nhánh',
        'edit_branches' => 'Sửa chi nhánh',
        'delete_branches' => 'Xóa chi nhánh',
        'manage_branch_settings' => 'Cài đặt chi nhánh',
        
        // Department Management
        'view_departments' => 'Xem phòng ban',
        'create_departments' => 'Tạo phòng ban',
        'edit_departments' => 'Sửa phòng ban',
        'delete_departments' => 'Xóa phòng ban',
        'assign_department_leaders' => 'Phân công trưởng phòng',
        
        // System Settings
        'view_system_settings' => 'Xem cài đặt hệ thống',
        'edit_system_settings' => 'Sửa cài đặt hệ thống',
        'manage_user_permissions' => 'Quản lý quyền người dùng',
        'backup_system' => 'Sao lưu hệ thống',
        'view_system_logs' => 'Xem nhật ký hệ thống',
        
        // Advanced Features
        'manage_integrations' => 'Quản lý tích hợp',
        'system_maintenance' => 'Bảo trì hệ thống',
        'financial_reports' => 'Báo cáo tài chính',
        'audit_logs' => 'Nhật ký kiểm toán'
    ];
    
    // Định nghĩa quyền mặc định cho từng role
    const ROLE_PERMISSIONS = [
        'owner' => [
            // Toàn quyền
            'view_dashboard', 'view_reports', 'export_reports',
            'view_tasks', 'create_tasks', 'edit_tasks', 'delete_tasks', 'complete_tasks',
            'view_kpi', 'edit_kpi', 'view_team_kpi', 'set_kpi_targets',
            'view_sales', 'create_sales', 'edit_sales', 'delete_sales', 'view_all_sales',
            'view_employees', 'create_employees', 'edit_employees', 'delete_employees', 'manage_employee_roles',
            'view_branches', 'create_branches', 'edit_branches', 'delete_branches', 'manage_branch_settings',
            'view_departments', 'create_departments', 'edit_departments', 'delete_departments', 'assign_department_leaders',
            'view_system_settings', 'edit_system_settings', 'manage_user_permissions', 'backup_system', 'view_system_logs',
            'manage_integrations', 'system_maintenance', 'financial_reports', 'audit_logs'
        ],
        'manager' => [
            // Quản lý chi nhánh
            'view_dashboard', 'view_reports', 'export_reports',
            'view_tasks', 'create_tasks', 'edit_tasks', 'delete_tasks', 'complete_tasks',
            'view_kpi', 'edit_kpi', 'view_team_kpi', 'set_kpi_targets',
            'view_sales', 'create_sales', 'edit_sales', 'view_all_sales',
            'view_employees', 'create_employees', 'edit_employees', 'manage_employee_roles',
            'view_branches', 'edit_branches',
            'view_departments', 'create_departments', 'edit_departments', 'assign_department_leaders',
            'view_system_settings'
        ],
        'team_leader' => [
            // Quản lý team
            'view_dashboard', 'view_reports',
            'view_tasks', 'create_tasks', 'edit_tasks', 'complete_tasks',
            'view_kpi', 'edit_kpi', 'view_team_kpi',
            'view_sales', 'create_sales', 'edit_sales',
            'view_employees', 'edit_employees',
            'view_departments'
        ],
        'employee' => [
            // Nhân viên cơ bản
            'view_dashboard',
            'view_tasks', 'complete_tasks',
            'view_kpi',
            'view_sales', 'create_sales'
        ]
    ];

    public function __construct($db) {
        $this->conn = $db;
    }

    public function hasPermission($userId, $permission) {
        // Lấy role của user
        $query = "SELECT role FROM users WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) return false;
        
        // Kiểm tra quyền theo role
        $rolePermissions = self::ROLE_PERMISSIONS[$user['role']] ?? [];
        
        // Kiểm tra quyền custom của user (nếu có)
        $customPermissions = $this->getUserCustomPermissions($userId);
        
        return in_array($permission, $rolePermissions) || in_array($permission, $customPermissions);
    }
    
    public function getUserPermissions($userId) {
        // Lấy role của user
        $query = "SELECT role FROM users WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) return [];
        
        // Lấy quyền theo role
        $rolePermissions = self::ROLE_PERMISSIONS[$user['role']] ?? [];
        
        // Lấy quyền custom
        $customPermissions = $this->getUserCustomPermissions($userId);
        
        return array_unique(array_merge($rolePermissions, $customPermissions));
    }
    
    public function getUserCustomPermissions($userId) {
        $query = "SELECT permission FROM user_permissions WHERE user_id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
    
    public function setUserPermission($userId, $permission, $granted = true) {
        if ($granted) {
            $query = "INSERT IGNORE INTO user_permissions (user_id, permission, granted_at) 
                      VALUES (:user_id, :permission, NOW())";
        } else {
            $query = "DELETE FROM user_permissions WHERE user_id = :user_id AND permission = :permission";
        }
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->bindParam(':permission', $permission);
        
        return $stmt->execute();
    }
    
    public function getAllPermissions() {
        return self::PERMISSIONS;
    }
    
    public function getRolePermissions($role) {
        return self::ROLE_PERMISSIONS[$role] ?? [];
    }
}
?>